/*****************************************************************************
*  LPC11U6x_E6x PWM Decode demo program
*
*  Use SCT timer capture and capture control features. It implements a
*  PWM decoder which measures the duty cycle of a PWM signal and determines
*  whether it is above (max_width) or below (min_width) a specific value.
*  The PWM signal frequency is assumed to be 10 kHz. Two output signals
*  (width_error and timeout) are included to indicate when the 10 kHz signal
*  has an error or is missing.
*****************************************************************************/
#include "board.h"

#define PWM_FREQUENCY           10000                      // PWM frequency in Hz
#define PWM_RESOLUTION_NS       1000                       // Timer resolution in ns
#define PWM_MIN_DUTY_PERCENT    25                         // Minimum allowed duty cycle in %
#define PWM_MAX_DUTY_PERCENT    70                         // Maximum allowed duty cycle in %

/* Derived constants */
#define SCT_PRESCALER           (((SystemCoreClock / 1000u) * PWM_RESOLUTION_NS) / 1000000u - 1u)

#define match_min_width         ((10000000u * PWM_MIN_DUTY_PERCENT) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))
#define match_max_width         ((10000000u * PWM_MAX_DUTY_PERCENT) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))
#define match_no_input          ((10000000u * 300                 ) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))


void SCT1_Init(void);


void SCT1_Init(void)
{

	Chip_SCT_Init(LPC_SCT1);							   				//enable clock for SCT0/1

	Chip_SCT_Config(LPC_SCT1,  SCT_CONFIG_32BIT_COUNTER | 				//Unified
							   SCT_CONFIG_AUTOLIMIT_L	);				//auto limit

	Chip_SCT_SetControl(LPC_SCT1, SCT_CTRL_PRE_L(SCT_PRESCALER));		// set pre-scaler

	LPC_SCT1->REGMODE_L      = 0x00000018;                 				// 3x MATCH, 2x CAPTURE used

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_0, match_max_width);				// match_max_width
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_0, match_max_width);

	Chip_SCT_SetMatchCount(LPC_SCT1,SCT_MATCH_1,match_min_width);					// match_min_width
	Chip_SCT_SetMatchReload(LPC_SCT1,SCT_MATCH_1,match_min_width );

	Chip_SCT_SetMatchCount(LPC_SCT1,SCT_MATCH_2,match_no_input);					// match_no_input
	Chip_SCT_SetMatchReload(LPC_SCT1,SCT_MATCH_2,match_no_input );


	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_0, ENABLE_ALL_STATES);		// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH2 |
                                                                        SCT_COMBMODE_MATCH ));	// related to match_no_input only

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_1, ENABLE_ALL_STATES);		// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) ( SCT_IOCOND_RAISE |
                                                                        SCT_COMBMODE_IO  )); 	// IN_0 rising edge condition only

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_2, ENABLE_STATE0);			// event 2 happens in state 0
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1  | 	// related to match_min_width
                                                                        SCT_COMBMODE_MATCH  |	  // match condition only
                                                                        SCT_STATELD_1		    | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_1		    ));	// new state is 1

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_3, ENABLE_STATE1);		    // event 3 happens in state 1
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_3, (CHIP_SCT_EVENTCTRL_T) ( SCT_IOCOND_FALL  	| 	// IN_0 falling edge
                                                                        SCT_COMBMODE_IO   |	  // IO condition only
                                                                        SCT_STATELD_1		  | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_0		  ));	// new state is 0

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_4, ENABLE_STATE1);		    // event 4 happens in state 1
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0  | 	// related to match_max_width
                                                                        SCT_COMBMODE_MATCH  |	  // match condition only
                                                                        SCT_STATELD_1		    | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_0		    ));	// new state is 0


	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_5, ENABLE_STATE0);		    // event 5 happens in state 0
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_5, (CHIP_SCT_EVENTCTRL_T) ( SCT_IOCOND_FALL |  	// IN_0 falling edge condition only
                                                                        SCT_COMBMODE_IO ));

	Chip_SCT_CapControl(LPC_SCT1, SCT_CAPCTRL_3 , SCT_EVT_0);				// event 1 is causing capture 3
	Chip_SCT_CapControl(LPC_SCT1, SCT_CAPCTRL_4 , (CHIP_SCT_EVENT_T)  ( SCT_EVT_3 |
                                                                      SCT_EVT_5 ));				// event 3 and 5 cause capture 4


	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_0, SCT_EVT_1);				// event 1 set   OUT0 (no timeout)
	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_0, SCT_EVT_0);			// event 0 clear OUT0 (timeout)

	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_1, SCT_EVT_3);				// event 3 set   OUT1 (no width error)
	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T) ( SCT_EVT_0 |
                                                                    SCT_EVT_5 ));			// event 0 and 5 clear OUT1 (width error)

	Chip_SCT_Output(LPC_SCT1, 0x03);        							// default set OUT0 and OUT1

	LPC_SCT1->LIMIT_L        = (1 << 0) | (1 << 1);        				// event 0 and 1 limit the timer

	Chip_SCT_EnableEventInt(LPC_SCT1, (CHIP_SCT_EVENT_T)  (	SCT_EVT_0 |
                                                          SCT_EVT_5 ));	   				// event 0 and 5 generate an irq


	NVIC_EnableIRQ(SCT0_1_IRQn);                             			// enable SCT0 interrupt

	Chip_SCT_ClearControl(LPC_SCT1,SCT_CTRL_HALT_L);	   				// unhalt it by clearing bit 2 of the CTRL register


}


void SCT0_1_IRQHandler(void)
{
	uint32_t status = Chip_SCT_GetEventFlag(LPC_SCT1);

	    if (status & 1)                                    		// EVENT 0: timeout
	    {
			/* New measurement result */
			;
	    }

	    if (status & (1 << 5))                             		// EVENT 5: width error
	    {
			/* Time-out (no signal) */
			;
	    }
	    Chip_SCT_ClearEventFlag(LPC_SCT1, (CHIP_SCT_EVENT_T)status);		   		// Acknowledge interrupts
}

int main (void)
{

	volatile static int i = 0;                           		// force counter to be placed into memory
	SystemCoreClockUpdate();
	Board_Init();

	Chip_Clock_SetSysClockDiv(4);

	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_IOCON);				// enable clock for IOCON


	Chip_IOCON_PinMuxSet(LPC_IOCON, 2, 8 , 0x00000081);				// P2_8  is SCT1_IN0  = PWM signal input
	Chip_IOCON_PinMuxSet(LPC_IOCON, 2,16 , 0x00000081);				// P2_16 is SCT1_OUT0 = timeout (red LED)
	Chip_IOCON_PinMuxSet(LPC_IOCON, 2,17 , 0x00000081);				// P2_17 is SCT1_OUT1 = width_error (green LED)

	SCT1_Init();                                   					// Initialize SCT


	while (1)                                              			// loop forever
	{
		/* The settings are chosen in % of duty cycle (assuming a PWM frequency of 10 kHz).
		       	 * (In a real world application you want to select better resolution!)
		       	 */
		        i = LPC_SCT1->CAP[4].U;
		        __WFI();
	}
}

